<?php
/**
 * GameCS-UA Telegram Bot by bot1 v1.1 — CS 1.6 Monitor + Stats + Moderation
 * 
 */

require_once __DIR__.'/config.php';
require_once __DIR__.'/SourceQuery/bootstrap.php';

use xPaw\SourceQuery\SourceQuery;

date_default_timezone_set('Europe/Kiev');

/* ========================= DB + Settings ========================= */

function db() {
  global $CFG;
  static $pdo=null;
  if($pdo) return $pdo;
  $dsn="mysql:host={$CFG['db']['host']};dbname={$CFG['db']['name']};charset=utf8mb4";
  $pdo=new PDO($dsn,$CFG['db']['user'],$CFG['db']['pass'],[
    PDO::ATTR_ERRMODE=>PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE=>PDO::FETCH_OBJ,
  ]);
  return $pdo;
}

function settings_load() {
  $st=db()->query("SELECT `key`,`value` FROM bot_settings");
  $out=[]; foreach($st as $r){ $out[$r->key]=$r->value; } return $out;
}
$S = settings_load();
$lang = $S['lang'] ?? ($CFG['default_lang'] ?? 'ru');

function esc($s){ return htmlspecialchars((string)$s,ENT_QUOTES|ENT_SUBSTITUTE,'UTF-8'); }

/* ========================= I18N tiny ========================= */

function T($key,$fallback){
  global $lang,$S;
  $ru = [
    'menu_title'=> $S['phrase_menu'] ?? "📋 <b>Главное меню</b>\nВыберите действие:",
    'btn_me'    => "🎮 Моя статистика",
    'btn_top'   => "🏆 Топ-игроки",
    'btn_info'  => "ℹ️ Инфо / Сервер",
    'btn_back'  => "⬅️ Назад",
    'btn_top_k' => "🔫 Убийства",
    'btn_top_a' => "💰 Бонусы",
    'btn_top_t' => "⏱ Время",
    'btn_top_o' => "⚔️ Общий рейтинг",
    'btn_help'  => "🆘 Помощь",
    'btn_online'=> "🟢 Онлайн",
  ];
  $ua = [
    'menu_title'=> "📋 <b>Головне меню</b>\nОберіть дію:",
    'btn_me'    => "🎮 Моя статистика",
    'btn_top'   => "🏆 Топ-гравці",
    'btn_info'  => "ℹ️ Інфо / Сервер",
    'btn_back'  => "⬅️ Назад",
    'btn_top_k' => "🔫 Вбивства",
    'btn_top_a' => "💰 Бонуси",
    'btn_top_t' => "⏱ Час",
    'btn_top_o' => "⚔️ Загальний рейтинг",
    'btn_help'  => "🆘 Допомога",
    'btn_online'=> "🟢 Онлайн",
  ];
  $dict = $lang==='ua' ? $ua : $ru;
  return $dict[$key] ?? $fallback;
}

/* ========================= Telegram HTTP ========================= */

function tg($method,$params=[]){
  global $CFG;
  $ch=curl_init("https://api.telegram.org/bot{$CFG['bot_token']}/$method");
  curl_setopt_array($ch,[
    CURLOPT_POST=>true,
    CURLOPT_POSTFIELDS=>$params,
    CURLOPT_RETURNTRANSFER=>true,
    CURLOPT_CONNECTTIMEOUT=>$CFG['http_timeout'] ?? 5,
    CURLOPT_TIMEOUT=>$CFG['http_timeout'] ?? 5,
  ]);
  $res=curl_exec($ch);
  curl_close($ch);
  return $res?json_decode($res,true):['ok'=>false,'description'=>'no response'];
}
function sendMessage($chat_id,$text,$reply_to=null,$markup=null){
  $p=['chat_id'=>$chat_id,'text'=>$text,'parse_mode'=>'HTML','disable_web_page_preview'=>true];
  if($reply_to) $p['reply_to_message_id']=$reply_to;
  if($markup)   $p['reply_markup']=json_encode($markup);
  return tg('sendMessage',$p);
}
function editMessage($chat_id,$mid,$text,$markup=null){
  $p=['chat_id'=>$chat_id,'message_id'=>$mid,'text'=>$text,'parse_mode'=>'HTML'];
  if($markup) $p['reply_markup']=json_encode($markup);
  return tg('editMessageText',$p);
}
function sendPhoto($chat_id,$photo,$caption,$reply_to=null,$markup=null){
  $p=['chat_id'=>$chat_id,'photo'=>$photo,'caption'=>$caption,'parse_mode'=>'HTML'];
  if($reply_to) $p['reply_to_message_id']=$reply_to;
  if($markup)   $p['reply_markup']=json_encode($markup);
  return tg('sendPhoto',$p);
}

/* ========================= Helpers ========================= */

function isHeadAdmin($tg_id){ global $CFG; return in_array((int)$tg_id,$CFG['head_admins']??[],true); }
function isChatAdmin($chat_id,$user_id){
  $r=tg('getChatMember',['chat_id'=>$chat_id,'user_id'=>$user_id]);
  if(!($r['ok']??false)) return false;
  $st=$r['result']['status']??'';
  return in_array($st,['administrator','creator'],true) || isHeadAdmin($user_id);
}

function value_k($v){ $v=(int)$v; return $v>=1000? (round($v/1000,1).'K') : (string)$v; }
function time_h($sec){ $sec=(int)$sec; return floor($sec/3600).' ч.'; }

/**
 * Буква разряда по вашей шкале (БЕЗ нормализации DB-скилла):
 * пороги: 0 60 75 85 100 115 130 140 150 165 180 195 210
 * ярлыки: L- L L+ M- M M+ H- H H+ P- P P+ G
 * Алгоритм: берём ПОСЛЕДНИЙ порог ≤ skill_db.
 */
function grade_from_db_skill($skill_db){
  $s = (int)$skill_db;
  if ($s < 0) $s = 0;
  $thresholds = [0, 60, 75, 85, 100, 115, 130, 140, 150, 165, 180, 195, 210];
  $labels     = ['L-','L','L+','M-','M','M+','H-','H','H+','P-','P','P+','G'];

  $idx = 0; $n = count($thresholds);
  for ($i=0; $i<$n; $i++){
    if ($s >= $thresholds[$i]) $idx = $i;
    else break;
  }
  return $labels[$idx];
}

function format_stats($row,$steam,$rank,$tg_user=null,$title='🎯 Ваша статистика'){
  $nick=esc($row->nick);
  $caption = "<b>{$title}</b>\n";
  if($rank) $caption.="🏅 <b>Место:</b> <code>$rank</code>\n";
  $caption.="👤 <b>Ник:</b> <code>{$nick}</code>\n";
  if($tg_user){
    $tgname = esc($tg_user['name']);
    $caption.="📱 <b>Telegram:</b> <a href='tg://user?id={$tg_user['id']}'>{$tgname}</a>\n";
  }
  $caption.=
    "🔫 <b>Убийства:</b> <code>".value_k($row->frags)."</code>\n".
    "💀 <b>Смерти:</b> <code>".value_k($row->deaths)."</code>\n".
    "🥏 <b>Хедшоты:</b> <code>".value_k($row->headshots)."</code>\n".
    "💥 <b>Урон:</b> <code>".value_k($row->damage)."</code>\n".
    "🎁 <b>Бонусы:</b> <code>".value_k($row->ar_anew)."</code>\n".
    "⌛ <b>Время:</b> <code>".time_h($row->gametime)."</code>\n";

  // Skill (DB) + буква из вашей шкалы
  if(isset($row->skill)){
    $letter = grade_from_db_skill((int)$row->skill);
    $caption.="📈 <b>Skill:</b> <code>".(int)$row->skill."</code>  (<b>{$letter}</b>)\n";
  }

  if($steam) $caption.="🌐 <b>Steam ID:</b> <code>".esc($steam)."</code>\n";
  return $caption;
}

/* ========================= Steam link helpers ========================= */

function getSteamByTg($tg){
  $st=db()->prepare("SELECT steamid FROM steam_links WHERE tg_id=:i");
  $st->execute([':i'=>$tg]);
  $r=$st->fetch();
  return $r?$r->steamid:null;
}
function setSteam($tg,$steam,$username=null){
  $st=db()->prepare("INSERT INTO steam_links(tg_id,steamid,username) VALUES(:i,:s,:u)
    ON DUPLICATE KEY UPDATE steamid=VALUES(steamid), username=VALUES(username)");
  $st->execute([':i'=>$tg,':s'=>$steam,':u'=>$username]);
}

/* ========================= Warn/Mute storage ========================= */

function warn_add($chat,$tg){
  $st=db()->prepare("INSERT INTO tg_warns(chat_id,tg_id,warns) VALUES(:c,:t,1)
    ON DUPLICATE KEY UPDATE warns=warns+1");
  $st->execute([':c'=>$chat,':t'=>$tg]);
}
function warn_get($chat,$tg){
  $st=db()->prepare("SELECT warns FROM tg_warns WHERE chat_id=:c AND tg_id=:t");
  $st->execute([':c'=>$chat,':t'=>$tg]);
  $r=$st->fetch(); return $r?(int)$r->warns:0;
}
function warn_dec($chat,$tg){
  $w=warn_get($chat,$tg); if($w<=0) return 0;
  $st=db()->prepare("UPDATE tg_warns SET warns=GREATEST(warns-1,0) WHERE chat_id=:c AND tg_id=:t");
  $st->execute([':c'=>$chat,':t'=>$tg]); return $w-1;
}
function mute_set($chat,$tg,$until){
  $st=db()->prepare("INSERT INTO tg_mutes(chat_id,tg_id,until_ts) VALUES(:c,:t,:u)
    ON DUPLICATE KEY UPDATE until_ts=:u");
  $st->execute([':c'=>$chat,':t'=>$tg,':u'=>$until]);
}
function mute_clear($chat,$tg){
  $st=db()->prepare("DELETE FROM tg_mutes WHERE chat_id=:c AND tg_id=:t");
  $st->execute([':c'=>$chat,':t'=>$tg]);
}

/* ========================= SourceQuery (cache 20s) ========================= */

define('CACHE_FILE', __DIR__.'/server_cache.json');
define('CACHE_TTL', 20);

function cache_get($key){
  if(!is_file(CACHE_FILE)) return null;
  $j=@json_decode(file_get_contents(CACHE_FILE),true);
  if(!$j || !isset($j[$key])) return null;
  if(time()-$j[$key]['ts']>CACHE_TTL) return null;
  return $j[$key]['data'];
}
function cache_set($key,$data){
  $j=is_file(CACHE_FILE)?@json_decode(file_get_contents(CACHE_FILE),true):[];
  $j[$key]=['ts'=>time(),'data'=>$data];
  @file_put_contents(CACHE_FILE,json_encode($j,JSON_UNESCAPED_UNICODE|JSON_PRETTY_PRINT));
}

function server_query($ip,$port,$want_players=true){
  $key="{$ip}:{$port}_".($want_players?'players':'info');
  if($c=cache_get($key)) return $c;

  $q=new SourceQuery();
  try{
    $q->Connect($ip,(int)$port,5,SourceQuery::GOLDSOURCE);
    $info=$q->GetInfo();

    // GetRules может падать — игнорируем ошибку
    $timeleft='Неизвестно';
    try{
      $rules=$q->GetRules();
      $tl = $rules['mp_timeleft'] ?? '0';
      if($tl!=='0' && $tl!=='') $timeleft=$tl;
    }catch(Exception $e){ $timeleft='Неизвестно'; }

    $data=[
      'name'=>$info['HostName']??'',
      'map'=>$info['Map']??'',
      'players'=>$info['Players']??0,
      'max'=>$info['MaxPlayers']??0,
      'timeleft'=>$timeleft
    ];
    if($want_players){
      $pl=[];
      try{ $pl=$q->GetPlayers(); }catch(Exception $e){ $pl=[]; }
      if(is_array($pl)) usort($pl,fn($a,$b)=>($b['Frags']<=>$a['Frags']));
      $data['list']=$pl;
    }
    cache_set($key,$data);
    return $data;
  }catch(Exception $e){
    return ['error'=>'Server query failed: '.$e->getMessage()];
  }finally{
    $q->Disconnect();
  }
}

/* ========================= Update ========================= */

$update = json_decode(file_get_contents('php://input'), true);
if(!$update) exit;

if(isset($update['message'])) {
  $m=$update['message'];
  $chat_id=$m['chat']['id'];
  $mid=$m['message_id'];
  $from_id=$m['from']['id'];
  $from_name=$m['from']['first_name'].(isset($m['from']['last_name'])?' '.$m['from']['last_name']:'');
  $text=$m['text'] ?? '';
  $reply=$m['reply_to_message'] ?? null;

  // Greetings
  if(isset($m['new_chat_members'])){
    $u=$m['new_chat_members'][0];
    $name=esc($u['first_name']);
    $w = ($lang==='ua') ? ($S['welcome_ua']??'👋 Ласкаво просимо, {name}!') : ($S['welcome_ru']??'👋 Добро пожаловать, {name}!');
    $w = str_replace('{name}', "<a href='tg://user?id={$u['id']}'>{$name}</a>", $w);
    sendMessage($chat_id,$w,$mid); exit;
  }
  if(isset($m['left_chat_member'])){
    $u=$m['left_chat_member'];
    sendMessage($chat_id,"👋 <b>Пока,</b> <a href='tg://user?id={$u['id']}'>".esc($u['first_name'])."</a>!",$mid); exit;
  }

  // Commands
  if($text && $text[0]==='/'){
    // normalize (убираем @username у бота)
    $cmd = strtolower(trim(preg_replace('~@[\w_]+~','',$text)));

    // /menu
    if($cmd==='/menu'){
      $kb = [
        'inline_keyboard' => [
          [ ['text'=>T('btn_me','Моя статистика'),'callback_data'=>'menu:me'],
            ['text'=>T('btn_top','Топ'),'callback_data'=>'menu:top'] ],
          [ ['text'=>T('btn_info','Инфо / Сервер'),'callback_data'=>'menu:info'] ],
          [ ['text'=>T('btn_online','Онлайн'),'callback_data'=>'menu:online'],
            ['text'=>T('btn_help','Помощь'),'callback_data'=>'menu:help'] ],
        ]
      ];
      sendMessage($chat_id, T('menu_title','Меню'), $mid, $kb); exit;
    }

    // /help
    if($cmd==='/help' || $cmd==='/top_serv'){
      sendMessage($chat_id, $S['phrase_help'] ?? 'Справка', $mid); exit;
    }

    // id
    if($cmd==='/myid'){
      sendMessage($chat_id,"👤 Ваш ID: <code>{$from_id}</code>",$mid); exit;
    }
    if($cmd==='/yuid' && $reply){
      $tid=$reply['from']['id'];
      sendMessage($chat_id,"👤 ID пользователя: <code>{$tid}</code>",$mid); exit;
    }

    // setsteam / adsteam
    if(preg_match('~^/setsteam\s+(.+)~ui',$cmd,$mm)){
      $steam=trim($mm[1]);

      if(!preg_match('~^(STEAM|VALVE)_~i',$steam)){
        sendMessage($chat_id,"❌ Укажите STEAM_ID вида <code>STEAM_1:0:123456</code>",$mid); exit;
      }
      // проверим наличие в csstats_players
      $st=db()->prepare("SELECT 1 FROM csstats_players WHERE authid=:s LIMIT 1");
      $st->execute([':s'=>$steam]); $ok=$st->fetchColumn();
      if(!$ok){ sendMessage($chat_id,"❌ Вас нет в базе сервера. Зайдите на сервер и попробуйте снова.",$mid); exit; }

      // сохранить привязку
      try{
        setSteam($from_id,$steam,$m['from']['username']??null);
        sendMessage($chat_id,"✅ Привязан Steam: <code>$steam</code>",$mid);
      }catch(Throwable $e){
        sendMessage($chat_id,"❌ Не удалось сохранить привязку: <code>".esc($e->getMessage())."</code>",$mid);
      }
      exit;
    }

    if(preg_match('~^/adsteam\s+(.+)~ui',$cmd,$mm)){
      if(!isChatAdmin($chat_id,$from_id)){ sendMessage($chat_id,"❌ Нет прав.",$mid); exit; }
      if(!$reply){ sendMessage($chat_id,"❗ Ответьте на сообщение пользователя.",$mid); exit; }
      $steam=trim($mm[1]);
      if(!preg_match('~^(STEAM|VALVE)_~i',$steam)){
        sendMessage($chat_id,"❌ Укажите STEAM_ID вида <code>STEAM_1:0:123456</code>",$mid); exit;
      }
      $st=db()->prepare("SELECT 1 FROM csstats_players WHERE authid=:s LIMIT 1"); $st->execute([':s'=>$steam]);
      if(!$st->fetchColumn()){ sendMessage($chat_id,"❌ Нет такого Steam в базе.",$mid); exit; }
      try{
        setSteam($reply['from']['id'],$steam,$reply['from']['username']??null);
        sendMessage($chat_id,"✅ Привязан Steam <code>$steam</code> для пользователя.",$mid);
      }catch(Throwable $e){
        sendMessage($chat_id,"❌ Не удалось сохранить привязку: <code>".esc($e->getMessage())."</code>",$mid);
      }
      exit;
    }

    // /me
    if($cmd==='/me'){
      $steam=getSteamByTg($from_id);
      if(!$steam){ sendMessage($chat_id,"❌ Привяжите Steam: <code>/setsteam STEAM_1:0:123456</code>",$mid); exit; }
      $st=db()->prepare("SELECT authid,nick,frags,deaths,headshots,damage,gametime,ar_anew,IFNULL(skill,0) AS skill FROM csstats_players WHERE authid=:s LIMIT 1");
      $st->execute([':s'=>$steam]); $r=$st->fetch();
      if(!$r){ sendMessage($chat_id,"❌ Игрок не найден.",$mid); exit; }
      // место
      $rank=null;
      $sql="SELECT
        (SELECT COUNT(*)+1 FROM csstats_players p2
          WHERE (p2.frags - p2.deaths) > (p1.frags - p1.deaths)
             OR ((p2.frags - p2.deaths) = (p1.frags - p1.deaths) AND p2.frags > p1.frags)
        ) as rank_pos
        FROM csstats_players p1 WHERE p1.authid = :a LIMIT 1";
      $pst=db()->prepare($sql); $pst->execute([':a'=>$steam]);
      if($x=$pst->fetch()) $rank=$x->rank_pos;

      $tguser=['id'=>$from_id,'name'=>$m['from']['username']?('@'.$m['from']['username']):$from_name];
      $title=$S['phrase_me_title']??'🎯 Ваша статистика';
      $text=format_stats($r,$steam,$rank,$tguser,$title);
      sendMessage($chat_id,$text,$mid); exit;
    }

    // /yu (reply)
    if($cmd==='/yu' && $reply){
      $tid=$reply['from']['id']; $steam=getSteamByTg($tid);
      if(!$steam){ sendMessage($chat_id,"❌ У пользователя нет привязки Steam.",$mid); exit; }
      $st=db()->prepare("SELECT authid,nick,frags,deaths,headshots,damage,gametime,ar_anew,IFNULL(skill,0) AS skill FROM csstats_players WHERE authid=:s LIMIT 1");
      $st->execute([':s'=>$steam]); $r=$st->fetch();
      if(!$r){ sendMessage($chat_id,"❌ Игрок не найден.",$mid); exit; }
      // место
      $rank=null;
      $pst=db()->prepare("SELECT
        (SELECT COUNT(*)+1 FROM csstats_players p2
          WHERE (p2.frags - p2.deaths) > (p1.frags - p1.deaths)
             OR ((p2.frags - p2.deaths) = (p1.frags - p1.deaths) AND p2.frags > p1.frags)
        ) as rank_pos
        FROM csstats_players p1 WHERE p1.authid = :a LIMIT 1");
      $pst->execute([':a'=>$steam]); if($x=$pst->fetch()) $rank=$x->rank_pos;

      $tguser=['id'=>$tid,'name'=>$reply['from']['username']?('@'.$reply['from']['username']):($reply['from']['first_name']??'User')];
      $text=format_stats($r,$steam,$rank,$tguser,'📊 Статистика игрока');
      sendMessage($chat_id,$text,$mid); exit;
    }

    // /play_Ник
    if(preg_match('~^/play_(.+)$~ui',$cmd,$mm)){
      $nick=trim($mm[1]);
      $st=db()->prepare("SELECT authid,nick,frags,deaths,headshots,damage,gametime,ar_anew,IFNULL(skill,0) AS skill FROM csstats_players WHERE LOWER(nick) LIKE LOWER(:n) ORDER BY frags DESC LIMIT 1");
      $st->execute([':n'=>"%$nick%"]); $r=$st->fetch();
      if(!$r){ sendMessage($chat_id,"❌ Игрок <b>".esc($nick)."</b> не найден.",$mid); exit; }
      // место
      $rank=null;
      $pst=db()->prepare("SELECT
        (SELECT COUNT(*)+1 FROM csstats_players p2
          WHERE (p2.frags - p2.deaths) > (p1.frags - p1.deaths)
             OR ((p2.frags - p2.deaths) = (p1.frags - p1.deaths) AND p2.frags > p1.frags)
        ) as rank_pos
        FROM csstats_players p1 WHERE p1.authid = :a LIMIT 1");
      $pst->execute([':a'=>$r->authid]); if($x=$pst->fetch()) $rank=$x->rank_pos;

      $text=format_stats($r,$r->authid,$rank,null,'📌 Найден игрок');
      sendMessage($chat_id,$text,$mid); exit;
    }

    // ТОПы
    $do_top = function($query,$header,$fmt){
      global $chat_id,$mid;
      $rs=db()->query($query)->fetchAll();
      if(!$rs){ sendMessage($chat_id,"Нет данных.",$mid); return; }
      $out = "<b>{$header}</b>\n<code>";
      $medals=["🥇","🥈","🥉","4️⃣","5️⃣","6️⃣","7️⃣","8️⃣","9️⃣","🔟"];
      foreach($rs as $i=>$r){
        $nick=str_pad(mb_substr($r->nick,0,18),18,' ');
        $val = $fmt==='time' ? time_h($r->gametime)
             : ($fmt==='raw' ? '' : str_pad(value_k($r->{$fmt}),6,' ',STR_PAD_LEFT));
        $out .= $fmt==='raw'
          ? "{$medals[$i]} {$nick}\n"
          : "{$medals[$i]} {$nick} | {$val}\n";
      }
      $out .= "</code>";
      sendMessage($chat_id,$out,$mid);
    };
    if($cmd==='/top_kill' || $cmd==='/top_kills'){
      $do_top("SELECT nick, frags FROM csstats_players WHERE frags>0 ORDER BY frags DESC LIMIT 10","🏆 Топ-10 по убийствам",'frags'); exit;
    }
    if($cmd==='/top_anew'){
      $do_top("SELECT nick, ar_anew FROM csstats_players WHERE ar_anew>0 ORDER BY ar_anew DESC LIMIT 10","🏆 Топ-10 по бонусам",'ar_anew'); exit;
    }
    if($cmd==='/top_time'){
      $do_top("SELECT nick, gametime FROM csstats_players WHERE gametime>0 ORDER BY gametime DESC LIMIT 10","🏆 Топ-10 по времени",'time'); exit;
    }
    if($cmd==='/top10' || $cmd==='/top'){
      $do_top("SELECT nick,(frags-deaths) AS score FROM csstats_players ORDER BY score DESC, frags DESC LIMIT 10","🏆 Общий топ",'raw'); exit;
    }

    // ONLINE + INFO
    if($cmd==='/online' || $cmd==='/info'){
      $ip=$S['server_ip']??'91.211.118.88'; $port=(int)($S['server_port']??'27055');
      $data=server_query($ip,$port,$cmd==='/online');
$up = !isset($data['error']);
$status = $up ? "🟢 Онлайн" : "🔴 Офлайн";

if(!$up){
  // показываем, что сервер офлайн и выходим
  $cap = "💻 <b>".($S['phrase_online']??'Состояние сервера')."</b> — <b>$status</b>\n";
  sendMessage($chat_id,$cap,$mid);
  exit;
}

$name=esc($data['name']); $map=esc($data['map']); $tl=esc($data['timeleft']);
$cap = " <b>".($S['phrase_online']??'Состояние сервера')."</b> — <b>$status</b>\n";
      if($name) $cap.="💫 <b>Сервер:</b> <code>{$name}</code>\n";
      $cap.="🌍 <b>IP:</b> <code>{$ip}:{$port}</code>\n".
            "🗺 <b>Карта:</b> <code>{$map}</code>\n".
            "⏳ <b>До смены:</b> <code>{$tl}</code>\n".
            "👥 <b>Онлайн:</b> <code>{$data['players']}/{$data['max']}</code>\n";
      if($cmd==='/online' && !empty($data['list'])){
        foreach($data['list'] as $i=>$p){
          $nm=esc($p['Name']?:'Player'); $fr=(int)$p['Frags'];
          $med=$i==0?'🥇':($i==1?'🥈':($i==2?'🥉':'🎮'));
          $cap.="$med <b>{$nm}</b> — <code>{$fr} фр.</code>\n";
        }
      }
      if(!empty($S['banner_url'])){
        sendPhoto($chat_id,$S['banner_url'],$cap,$mid);
      } else {
        sendMessage($chat_id,$cap,$mid);
      }
      exit;
    }

    // STATUS
    if($cmd==='/status'){
      $info=tg('getChat',['chat_id'=>$chat_id]);
      $cnt=tg('getChatMemberCount',['chat_id'=>$chat_id]);
      $ads=tg('getChatAdministrators',['chat_id'=>$chat_id]);
      if(($info['ok']??false)&&($cnt['ok']??false)&&($ads['ok']??false)){
        $title=$info['result']['title']??'';
        $admins=array_map(function($a){
          return $a['user']['username']?('@'.$a['user']['username']):($a['user']['first_name']??'');
        },$ads['result']);
        $txt="📊 <b>Группа:</b> ".esc($title)."\n".
             "👥 <b>Участников:</b> <code>{$cnt['result']}</code>\n".
             "🔧 <b>Админы:</b> ".esc(implode(', ',$admins));
        sendMessage($chat_id,$txt,$mid);
      } else sendMessage($chat_id,"❗ Ошибка получения информации.",$mid);
      exit;
    }

    // MODERATION (warn/unwarn/ban/unban/mute/unmute/silence)
    if(preg_match('~^/(warn|unwarn|ban|unban|mute|unmute|silence)(?:\s+(.+))?~',$cmd,$mm)){
      $action=$mm[1]; $arg=trim($mm[2]??'');
      if(!isChatAdmin($chat_id,$from_id)){ sendMessage($chat_id,"❌ Нет прав.",$mid); exit; }

      if($action==='silence'){
        if(!in_array($arg,['on','off'],true)){ sendMessage($chat_id,"Используйте: <code>/silence on</code> или <code>/silence off</code>",$mid); exit; }
        $allow = $arg==='off';
        tg('setChatPermissions',[
          'chat_id'=>$chat_id,
          'permissions'=>json_encode([
            'can_send_messages'=>$allow,
            'can_send_media_messages'=>$allow,
            'can_send_polls'=>$allow,
            'can_send_other_messages'=>$allow,
            'can_add_web_page_previews'=>$allow
          ])
        ]);
        sendMessage($chat_id, $allow?'🔊 Тихий режим выключен.':'🔇 Тихий режим включен.',$mid);
        exit;
      }

      // определить цель
      if($reply){
        $tid=$reply['from']['id'];
        $tname = $reply['from']['username']?('@'.$reply['from']['username']):($reply['from']['first_name']??'User');
      } else if(ctype_digit($arg)){
        $tid=$arg; $tname='Пользователь';
      } else {
        sendMessage($chat_id,"Ответьте на сообщение пользователя либо укажите его ID.",$mid); exit;
      }
      // запрет драк с админами
      if(!isHeadAdmin($from_id)){
        $tm=tg('getChatMember',['chat_id'=>$chat_id,'user_id'=>$tid]);
        $tst=$tm['result']['status']??'';
        if(in_array($tst,['administrator','creator'],true)){
          sendMessage($chat_id,"🛑 Действия над админами запрещены.",$mid); exit;
        }
      }

      switch($action){
        case 'warn':
          warn_add($chat_id,$tid);
          $w=warn_get($chat_id,$tid);
          if($w>=3){
            tg('banChatMember',['chat_id'=>$chat_id,'user_id'=>$tid]);
            sendMessage($chat_id,"🚫 {$tname} исключён за 3 предупреждения.",$mid);
            db()->prepare("DELETE FROM tg_warns WHERE chat_id=:c AND tg_id=:t")->execute([':c'=>$chat_id,':t'=>$tid]);
          } else {
            sendMessage($chat_id,"⚠️ {$tname} получил предупреждение: <code>{$w}/3</code>.",$mid);
          }
        break;

        case 'unwarn':
          $left=warn_dec($chat_id,$tid);
          sendMessage($chat_id,"✅ Снято предупреждение. Осталось: <code>{$left}</code>.",$mid);
        break;

        case 'ban':
          tg('banChatMember',['chat_id'=>$chat_id,'user_id'=>$tid]);
          sendMessage($chat_id,"🚫 {$tname} забанен.",$mid);
        break;

        case 'unban':
          tg('unbanChatMember',['chat_id'=>$chat_id,'user_id'=>$tid,'only_if_banned'=>true]);
          sendMessage($chat_id,"✅ {$tname} разбанен.",$mid);
        break;

        case 'mute':
          if(!$arg || !preg_match('~\d+~',$arg)){ sendMessage($chat_id,"Укажите минуты: <code>/mute 15</code>",$mid); exit; }
          $min=(int)$arg;
          $until=time()+$min*60;
          tg('restrictChatMember',[
            'chat_id'=>$chat_id,'user_id'=>$tid,'until_date'=>$until,
            'permissions'=>json_encode(['can_send_messages'=>false,'can_send_media_messages'=>false,'can_send_polls'=>false,'can_send_other_messages'=>false,'can_add_web_page_previews'=>false])
          ]);
          mute_set($chat_id,$tid,$until);
          sendMessage($chat_id,"🔇 {$tname} замучен на {$min} мин.",$mid);
        break;

        case 'unmute':
          tg('restrictChatMember',[
            'chat_id'=>$chat_id,'user_id'=>$tid,
            'permissions'=>json_encode(['can_send_messages'=>true,'can_send_media_messages'=>true,'can_send_polls'=>true,'can_send_other_messages'=>true,'can_add_web_page_previews'=>true])
          ]);
          mute_clear($chat_id,$tid);
          sendMessage($chat_id,"✅ {$tname} размучен.",$mid);
        break;
      }
      exit;
    }

    // неизвестная команда — /help
    sendMessage($chat_id, $S['phrase_help'] ?? 'Справка', $mid); exit;
  }
}

/* ========================= Callbacks (меню) ========================= */

if(isset($update['callback_query'])){
  $cb=$update['callback_query'];
  $chat_id=$cb['message']['chat']['id'];
  $mid=$cb['message']['message_id'];
  $from_id=$cb['from']['id'];
  $data=$cb['data'] ?? '';

  if($data==='menu:top'){
    $kb=['inline_keyboard'=>[
      [ ['text'=>T('btn_top_k','Убийства'),'callback_data'=>'top:kills'],
        ['text'=>T('btn_top_a','Бонусы'),'callback_data'=>'top:anew'] ],
      [ ['text'=>T('btn_top_t','Время'),'callback_data'=>'top:time'],
        ['text'=>T('btn_top_o','Общий рейтинг'),'callback_data'=>'top:overall'] ],
      [ ['text'=>T('btn_back','Назад'),'callback_data'=>'menu:root'] ]
    ]];
    editMessage($chat_id,$mid,T('menu_title','Меню'),$kb); exit;
  }

  if($data==='menu:me'){
    $steam=getSteamByTg($from_id);
    if(!$steam){ tg('answerCallbackQuery',['callback_query_id'=>$cb['id'],'text'=>'Привяжите Steam: /setsteam ...', 'show_alert'=>true]); exit; }
    $st=db()->prepare("SELECT authid,nick,frags,deaths,headshots,damage,gametime,ar_anew,IFNULL(skill,0) AS skill FROM csstats_players WHERE authid=:s LIMIT 1");
    $st->execute([':s'=>$steam]); $r=$st->fetch();
    if(!$r){ tg('answerCallbackQuery',['callback_query_id'=>$cb['id'],'text'=>'Игрок не найден', 'show_alert'=>true]); exit; }

    $pst=db()->prepare("SELECT
      (SELECT COUNT(*)+1 FROM csstats_players p2
        WHERE (p2.frags - p2.deaths) > (p1.frags - p1.deaths)
           OR ((p2.frags - p2.deaths) = (p1.frags - p1.deaths) AND p2.frags > p1.frags)
      ) as rank_pos
      FROM csstats_players p1 WHERE p1.authid = :a LIMIT 1");
    $pst->execute([':a'=>$steam]); $rank=null; if($x=$pst->fetch()) $rank=$x->rank_pos;

    $text=format_stats($r,$steam,$rank,['id'=>$from_id,'name'=>$cb['from']['username']?('@'.$cb['from']['username']):$cb['from']['first_name']], $S['phrase_me_title']??'🎯 Ваша статистика');
    editMessage($chat_id,$mid,$text); exit;
  }

  if($data==='menu:info'){
    $ip=$S['server_ip']??'91.211.118.88'; $port=(int)($S['server_port']??'27055');
    $d=server_query($ip,$port,false);
$up = !isset($d['error']);
$status = $up ? "🟢 Онлайн" : "🔴 Офлайн";

if(!$up){
  $cap="💻 <b>".($S['phrase_online']??'Состояние сервера')."</b> — <b>$status</b>\n";
  editMessage($chat_id,$mid,$cap);
  exit;
}
$name=esc($d['name']); $map=esc($d['map']); $tl=esc($d['timeleft']);
$cap=" <b>".($S['phrase_online']??'Состояние сервера')."</b> — <b>$status</b>\n";
    if($name) $cap.="💫 <b>Сервер:</b> <code>{$name}</code>\n";
    $cap.="🌍 <b>IP:</b> <code>{$ip}:{$port}</code>\n".
          "🗺 <b>Карта:</b> <code>{$map}</code>\n".
          "⏳ <b>До смены:</b> <code>{$tl}</code>\n";
    editMessage($chat_id,$mid,$cap); exit;
  }

  // 🔘 Помощь
  if($data==='menu:help'){
    $txt = $S['phrase_help'] ?? 'Справка';
    $kb=['inline_keyboard'=>[
      [ ['text'=>T('btn_back','Назад'),'callback_data'=>'menu:root'] ]
    ]];
    editMessage($chat_id,$mid,$txt,$kb); exit;
  }

  // 🔘 Онлайн (с игроками)
  if($data==='menu:online'){
    $ip=$S['server_ip']??'91.211.118.88'; $port=(int)($S['server_port']??'27055');
    $data=server_query($ip,$port,true);
$up = !isset($data['error']);
$status = $up ? "🟢 Онлайн" : "🔴 Офлайн";
if(!$up){
  $cap = "🎮 <b>".($S['phrase_online']??'Состояние сервера')."</b> — <b>$status</b>\n";
  $kb=['inline_keyboard'=>[[['text'=>T('btn_back','Назад'),'callback_data'=>'menu:root']]]];
  editMessage($chat_id,$mid,$cap,$kb);
  exit;
}
$name=esc($data['name']); $map=esc($data['map']); $tl=esc($data['timeleft']);
$cap = "🎮 <b>".($S['phrase_online']??'Состояние сервера')."</b> — <b>$status</b>\n";
    if($name) $cap.="🏷 <b>Сервер:</b> <code>{$name}</code>\n";
    $cap.="🌍 <b>IP:</b> <code>{$ip}:{$port}</code>\n".
          "🗺 <b>Карта:</b> <code>{$map}</code>\n".
          "⏳ <b>До смены:</b> <code>{$tl}</code>\n".
          "👥 <b>Онлайн:</b> <code>{$data['players']}/{$data['max']}</code>\n";
    if(!empty($data['list'])){
      foreach($data['list'] as $i=>$p){
        $nm=esc($p['Name']?:'Player'); $fr=(int)$p['Frags'];
        $med=$i==0?'🥇':($i==1?'🥈':($i==2?'🥉':'🎮'));
        $cap.="$med <b>{$nm}</b> — <code>{$fr} фр.</code>\n";
      }
    }
    $kb=['inline_keyboard'=>[
      [ ['text'=>T('btn_back','Назад'),'callback_data'=>'menu:root'] ]
    ]];
    editMessage($chat_id,$mid,$cap,$kb); exit;
  }

  if($data==='menu:root'){
    $kb = [
      'inline_keyboard' => [
        [ ['text'=>T('btn_me','Моя статистика'),'callback_data'=>'menu:me'],
          ['text'=>T('btn_top','Топ-игроки'),'callback_data'=>'menu:top'] ],
        [ ['text'=>T('btn_info','Инфо / Сервер'),'callback_data'=>'menu:info'] ],
        [ ['text'=>T('btn_online','Онлайн'),'callback_data'=>'menu:online'],
          ['text'=>T('btn_help','Помощь'),'callback_data'=>'menu:help'] ],
      ]
    ];
    editMessage($chat_id,$mid,T('menu_title','Меню'),$kb); exit;
  }

  // топы из меню
  if(strpos($data,'top:')===0){
    $mode=substr($data,4);
    $map=[
      'kills'=>["SELECT nick, frags f FROM csstats_players WHERE frags>0 ORDER BY frags DESC LIMIT 10","🏆 Топ-10 по убийствам",'f'],
      'anew' =>["SELECT nick, ar_anew f FROM csstats_players WHERE ar_anew>0 ORDER BY ar_anew DESC LIMIT 10","🏆 Топ-10 по бонусам",'f'],
      'time' =>["SELECT nick, gametime f FROM csstats_players WHERE gametime>0 ORDER BY gametime DESC LIMIT 10","🏆 Топ-10 по времени",'time'],
      'overall'=>["SELECT nick,(frags-deaths) f FROM csstats_players ORDER BY f DESC, frags DESC LIMIT 10","🏆 Общий топ",'raw'],
    ];
    if(!isset($map[$mode])) exit;
    [$q,$hdr,$fmt]=$map[$mode];

    $rs=db()->query($q)->fetchAll();
    $out="<b>{$hdr}</b>\n<code>";
    $med=["🥇","🥈","🥉","4️⃣","5️⃣","6️⃣","7️⃣","8️⃣","9️⃣","🔟"];
    foreach($rs as $i=>$r){
      $nick=str_pad(mb_substr($r->nick,0,18),18,' ');
      if($fmt==='raw'){
        $out.="{$med[$i]} {$nick}\n";
      } elseif($fmt==='time'){
        $out.="{$med[$i]} {$nick} | ".time_h($r->f)."\n";
      } else {
        $out.="{$med[$i]} {$nick} | ".str_pad(value_k($r->f),6,' ',STR_PAD_LEFT)."\n";
      }
    }
    $out.="</code>";
    $kb=['inline_keyboard'=>[
      [ ['text'=>T('btn_back','Назад'),'callback_data'=>'menu:top'] ]
    ]];
    editMessage($chat_id,$mid,$out,$kb); exit;
  }
}
